<?php
/**
 * Elementor Widget
 * @package bizmaster
 * @since 1.0.0
 */

namespace Elementor;

class Bizmaster_Service_Slider_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-service-slider-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Service Slider', 'bizmaster-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
		$this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control('read-btn', [
            'label' => esc_html__('Read More', 'bizmaster-core'),
            'type' => Controls_Manager::TEXT,
            'description' => esc_html__('enter read button text', 'bizmaster-core'),
            'default' => esc_html__('Read More', 'bizmaster-core'),
        ]);

        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'bizmaster-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many post you want in slider, enter -1 for unlimited post.')
        ]);

        $this->add_control('category', [
            'label' => esc_html__('Category', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => bizmaster()->get_terms_names('service-category', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'bizmaster-core'),
        ]);

        $this->add_control('order', [
            'label' => esc_html__('Order', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'bizmaster-core'),
                'DESC' => esc_html__('Descending', 'bizmaster-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'bizmaster-core'),
                'title' => esc_html__('Title', 'bizmaster-core'),
                'date' => esc_html__('Date', 'bizmaster-core'),
                'rand' => esc_html__('Random', 'bizmaster-core'),
                'comment_count' => esc_html__('Most Comments', 'bizmaster-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

		$this->add_control(
            'items',
            [
                'label' => esc_html__('Slides To Show', 'bizmaster-core'),
                'type' => Controls_Manager::NUMBER,
                'description' => esc_html__('you can set how many item show in slider', 'bizmaster-core'),
                'default' => '4',
            ]
        );

		$this->add_control(
            'dots',
            [
                'label' => esc_html__('Dots', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

		$this->add_control(
            'center',
            [
                'label' => esc_html__('Center', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),

            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'bizmaster-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'bizmaster-core'),
            ]
        );

        $this->add_control(
            'autoplaytimeout',
            [
                'label' => esc_html__('Autoplay Timeout', 'bizmaster-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 500,
                        'max' => 10000,
                        'step' => 100,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5000,
                ],
                'size_units' => ['px'],
                'condition' => array(
                    'autoplay' => 'yes'
                )
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'service_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .service-box_title a" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .service-box_title a:hover" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'bizmaster-core'),
            'name' => 'title_typography',
            'description' => esc_html__('Title Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .service-box_title a"
        ]);

		$this->add_control('content_color', [
            'label' => esc_html__('Content Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .service-box_text" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Content Typography', 'bizmaster-core'),
            'name' => 'content_typography',
            'description' => esc_html__('Content Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .service-box_text"
        ]);

		$this->add_control('slick_dots_bg_color', [
            'label' => esc_html__('Slick Dots BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
			'default' => '#54595F',
            'selectors' => [
                "{{WRAPPER}} .slick-dots" => "background-color: {{VALUE}}",
            ]
        ]);

        $this->add_control('slick_active_dot_bg_color', [
            'label' => esc_html__('Slick Active Dot BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .slick-dots .slick-active button" => "background-color: {{VALUE}}",
            ]
        ]);

        $this->end_controls_section();
    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
		$rand_numb = rand(333, 999999999);

		//slider settings
        $slider_settings = [
			"dots" => esc_attr($settings['dots']),
			"loop" => esc_attr($settings['loop']),
			"items" => esc_attr($settings['items'] ? $settings['items'] : 1),
            "center" => esc_attr($settings['center']),
            "autoplay" => esc_attr($settings['autoplay']),
            "autoplaytimeout" => esc_attr(!empty($settings['autoplaytimeout']['size']) ? $settings['autoplaytimeout']['size'] : 0)
		];

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];

		//setup query
        $args = array(
            'post_type' => 'service',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'service-category',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>
            <div id="service-slider-<?php echo esc_attr($rand_numb); ?>" data-settings='<?php echo json_encode($slider_settings); ?>' class="row gx-30 service-slider2">
                <?php
                while ($post_data->have_posts()) : $post_data->the_post();
					$post_id = get_the_ID();
					$service_single_meta_data = get_post_meta($post_id, 'bizmaster_service_options', true);
					$service_icon = isset($service_single_meta_data['service_icon']) && !empty($service_single_meta_data['service_icon']) ? $service_single_meta_data['service_icon'] : '';
					$service_details = isset($service_single_meta_data['service_content']) && !empty($service_single_meta_data['service_content']) ? $service_single_meta_data['service_content'] : '';
                    ?>
                    <div class="col-xl-3 col-lg-4 col-md-6">
                        <div class="service-box mt-5">
                            <div class="service-box_content">
                                <div class="service-box_icon">
									<?php
										if(!empty($service_icon['url'])) : ?>
											<img src="<?php echo esc_url($service_icon['url']); ?>" alt="<?php echo esc_attr('Service Icon', 'bizmaster-core'); ?>">
									<?php endif; ?>
                                </div>
                                <h4 class="service-box_title h5"><a href="<?php the_permalink(); ?>"><?php echo get_the_title(); ?></a></h4>
                                <p class="service-box_text"><?php echo $service_details; ?></p>
								<?php if(!empty($settings['read-btn'])) { ?>
									<a href="<?php the_permalink(); ?>" class="global-btn style-border"><?php echo esc_html($settings['read-btn']); ?></a>
								<?php } ?>
							</div>
                        </div>
                    </div>
                <?php endwhile; ?>
            </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Service_Slider_Widget());
